package com.smartgwt.sample.showcase.client.portal;

import com.google.gwt.core.client.GWT;
import com.smartgwt.client.bean.BeanFactory;
import com.smartgwt.client.data.DSCallback;
import com.smartgwt.client.data.DSRequest;
import com.smartgwt.client.data.DSResponse;
import com.smartgwt.client.data.DataSource;
import com.smartgwt.client.data.Record;
import com.smartgwt.client.tools.EditPane;
import com.smartgwt.client.tools.ListPalette;
import com.smartgwt.client.tools.PaletteNode;
import com.smartgwt.client.types.Alignment;
import com.smartgwt.client.types.ListGridEditEvent;
import com.smartgwt.client.types.SelectionStyle;
import com.smartgwt.client.types.Visibility;
import com.smartgwt.client.util.JSOHelper;
import com.smartgwt.client.widgets.Button;
import com.smartgwt.client.widgets.Canvas;
import com.smartgwt.client.widgets.events.ClickEvent;
import com.smartgwt.client.widgets.events.ClickHandler;
import com.smartgwt.client.widgets.grid.ListGrid;
import com.smartgwt.client.widgets.grid.ListGridField;
import com.smartgwt.client.widgets.grid.ListGridRecord;
import com.smartgwt.client.widgets.grid.events.CellContextClickEvent;
import com.smartgwt.client.widgets.grid.events.CellContextClickHandler;
import com.smartgwt.client.widgets.grid.events.RecordDoubleClickEvent;
import com.smartgwt.client.widgets.grid.events.RecordDoubleClickHandler;
import com.smartgwt.client.widgets.layout.HLayout;
import com.smartgwt.client.widgets.layout.LayoutSpacer;
import com.smartgwt.client.widgets.layout.PortalLayout;
import com.smartgwt.client.widgets.layout.VLayout;
import com.smartgwt.client.widgets.menu.Menu;
import com.smartgwt.client.widgets.menu.MenuItem;
import com.smartgwt.client.widgets.menu.MenuItemSeparator;
import com.smartgwt.client.widgets.menu.events.MenuItemClickEvent;
import com.smartgwt.client.widgets.tab.Tab;
import com.smartgwt.client.widgets.tab.TabSet;
import com.smartgwt.sample.showcase.client.PanelFactory;
import com.smartgwt.sample.showcase.client.ShowcasePanel;
import com.smartgwt.sample.showcase.client.SourceEntity;

public class PortalDashboardSample extends ShowcasePanel {

    private static final String DESCRIPTION =
        "<p>With the Tools framework, you can create dashboards of portlets."
        + "This example uses a list of pre-configured grid portlets that can be viewed or edited.</p>"
        + "<p>Double-click a pre-configured dashboard to view the saved portal layout consisting of"
        + "one or more grids. To make changes, select the dashboard and click Edit button."
        + "Change the criteria, field order or size, sort order, highlights,"
        + "or grouping to see how these properties are persisted.</p>"
        + "<p>Select a dashboard and click Clone button to generate another copy for experimentation. "
        + "Rename the new dashboard by right-clicking on the record and choosing Rename option.</p>";

    private ListGrid dashboardsList;
    private TabSet selector;
    private EditPane editPane;
    private HLayout editToolbar;
    private Record currentRecord;

    private PaletteNode initialPortalPaletteNode;

    private static boolean enabledReflection = false;

    public static class Factory implements PanelFactory {
        private String id;

        public ShowcasePanel create() {
        	PortalDashboardSample panel = new PortalDashboardSample();
            id = panel.getID();
            return panel;
        }

        public String getID() {
            return id;
        }

        public String getDescription() {
            return DESCRIPTION;
        }
    }

    public interface MetaFactory extends BeanFactory.MetaFactory {
        BeanFactory<Canvas> getCanvasBeanFactory();
        BeanFactory<PortalGrid> getPortalGridBeanFactory();
    }

    private static void enableReflection() {
        if (!enabledReflection) {
            GWT.create(MetaFactory.class);
            GWT.create(PortalGrid.MetaFactory.class);
            enabledReflection = true;
        }
    }

    public Canvas getViewPanel() {
        enableReflection();

        // Palette Node used to create a default portal
        PortalLayout defaults = new PortalLayout();
        defaults.setWidth100();
        defaults.setHeight100();
        defaults.setCanResizePortlets(true);
        initialPortalPaletteNode = new PaletteNode();
        initialPortalPaletteNode.setType("PortalLayout");
        initialPortalPaletteNode.setCanvasDefaults(defaults);

        ListPalette palette = createPalette();

        dashboardsList = createDashboardsList();
        HLayout dashboardsToolbar = createDashboardsToolbar();
        VLayout dashboardsPane = new VLayout();
        dashboardsPane.setMembers(dashboardsList, dashboardsToolbar);

        selector = new TabSet();
		Tab tab = new Tab("Dashboards");
		tab.setPane(dashboardsPane);
		selector.addTab(tab);
		tab = new Tab("Palette");
		tab.setPane(palette);
		tab.setDisabled(true);
		selector.addTab(tab);

		editPane = new EditPane();
		editPane.setBorder("1px solid black");
		editPane.setVisibility(Visibility.HIDDEN);
		JSOHelper.setAttribute(editPane.getConfig(), "editMode", false);

		// Make the new editPane the default Edit Context for the palette,
		// to support double-clicking on components in the palette to create them
		palette.setDefaultEditContext(editPane);
		editPane.setDefaultPalette(palette);

		// Add a PortalLayout to the editPane
		editPane.addFromPaletteNode(initialPortalPaletteNode);

		editToolbar = createEditToolbar();

        VLayout rightPane = new VLayout();
        rightPane.setMembers(editPane, editToolbar);

        // Layout for the example
        HLayout layout = new HLayout();
        layout.setWidth("100%");
        layout.setHeight("100%");
        layout.setMembersMargin(20);
        layout.setMembers(selector, rightPane);
        return layout;
    }

    private ListGrid createDashboardsList() {
    	final PortalDashboardSample dashboard = this;

    	final ListGrid grid = new ListGrid();
    	grid.setDataSource(DataSource.get("dashboards"));
    	grid.setAutoFetchData(true);
    	grid.setSelectionType(SelectionStyle.SINGLE);
    	grid.setSortField("description");
    	grid.setLeaveScrollbarGap(false);

	    // Allow edit of portal description (via context menu)
    	grid.setCanEdit(true);
    	grid.setEditEvent(ListGridEditEvent.NONE);
	    // And removal of dashboards
    	grid.setCanRemoveRecords(true);

    	grid.setFields(new ListGridField("description"));

    	grid.addRecordDoubleClickHandler(new RecordDoubleClickHandler() {
			@Override
			public void onRecordDoubleClick(RecordDoubleClickEvent event) {
				dashboard.clearCurrentDashboard();
				dashboard.viewDashboard();
			}
		});

    	final Menu contextMenu = new Menu();
    	grid.addCellContextClickHandler(new CellContextClickHandler() {
			@Override
			public void onCellContextClick(CellContextClickEvent event) {
				MenuItem renameItem = new MenuItem("Rename");
				renameItem.setAttribute("rowNum", event.getRowNum());
				renameItem.addClickHandler(new com.smartgwt.client.widgets.menu.events.ClickHandler() {
					@Override
					public void onClick(MenuItemClickEvent event) {
						int rowNum = event.getItem().getAttributeAsInt("rowNum");
						((ListGrid)event.getTarget()).startEditing(rowNum);
					}
				});
				MenuItem separator = new MenuItemSeparator();
				MenuItem editItem = new MenuItem("Edit");
				editItem.addClickHandler(new com.smartgwt.client.widgets.menu.events.ClickHandler() {
					@Override
					public void onClick(MenuItemClickEvent event) {
						dashboard.clearCurrentDashboard();
						dashboard.editDashboard();
					}
				});
				MenuItem cloneItem = new MenuItem("Clone");
				cloneItem.addClickHandler(new com.smartgwt.client.widgets.menu.events.ClickHandler() {
					@Override
					public void onClick(MenuItemClickEvent event) {
						dashboard.cloneDashboard();
					}
				});

				contextMenu.setItems(renameItem, separator, editItem, cloneItem);
				contextMenu.setTarget(grid);
				contextMenu.showContextMenu();

				// return false to kill the standard context menu
				event.cancel();
			}
		});
    	return grid;
    }

    private HLayout createDashboardsToolbar() {
    	final PortalDashboardSample dashboard = this;

    	Button viewButton = new Button("View");
    	viewButton.setAutoFit(true);
    	viewButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				dashboard.clearCurrentDashboard();
				dashboard.viewDashboard();
			}
		});

    	Button editButton = new Button("Edit");
    	editButton.setAutoFit(true);
    	editButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				dashboard.clearCurrentDashboard();
				dashboard.editDashboard();
			}
		});

    	Button newButton = new Button("New");
    	newButton.setAutoFit(true);
    	newButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				dashboard.newDashboard();
			}
		});

    	Button cloneButton = new Button("Clone");
    	cloneButton.setAutoFit(true);
    	cloneButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				dashboard.cloneDashboard();
			}
		});

    	HLayout toolbar = new HLayout();
    	toolbar.setHeight(30);
    	toolbar.setMembersMargin(10);
    	toolbar.setDefaultLayoutAlign(Alignment.CENTER);
    	toolbar.addMembers(new LayoutSpacer(), viewButton, editButton, newButton, cloneButton);

    	return toolbar;
    }
    
	private static ListPalette createPalette() {
		// The ListPalette contains components available
		// for use, with default settings.
		ListPalette palette = new ListPalette();
		palette.setLeaveScrollbarGap(false);
		palette.setFields(new ListGridField("title", "Component"));

		palette.setData(new PaletteNode[] {
			createPaletteNode("Animals", "animals"), 
			createPaletteNode("Supply Categories", "supplyCategory"), 
			createPaletteNode("Supply Items", "supplyItem")
		});

		return palette;
	}

	private static PaletteNode createPaletteNode(String title, String dataSource) {
		ListGrid defaults = new ListGrid();
        defaults.setDataSource(DataSource.get(dataSource));
        defaults.setAutoFetchData(true);
        defaults.setShowFilterEditor(true);
        PaletteNode node = new PaletteNode();
        node.setType("com.smartgwt.sample.showcase.client.portal.PortalGrid");
        node.setTitle(title);
        node.setCanvasDefaults(defaults);

        return node;
	}

    private HLayout createEditToolbar() {
    	final PortalDashboardSample dashboard = this;

    	Button saveButton = new Button("Save");
    	saveButton.setAutoFit(true);
    	saveButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				dashboard.saveDashboard();
			}
		});

    	Button discardButton = new Button("Discard changes");
    	discardButton.setAutoFit(true);
    	discardButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				dashboard.refreshDashboard();
			}
		});

    	HLayout toolbar = new HLayout();
    	toolbar.setHeight(30);
    	toolbar.setMembersMargin(10);
    	toolbar.setDefaultLayoutAlign(Alignment.CENTER);
    	toolbar.setVisibility(Visibility.HIDDEN);
    	toolbar.addMembers(new LayoutSpacer(), saveButton, discardButton);

    	return toolbar;
    }

    private void clearCurrentDashboard() {
    	editPane.destroyAll();
    	editPane.hide();
    	editToolbar.hide();
    }

    private void editDashboard() {
    	ListGridRecord record = dashboardsList.getSelectedRecord();
    	if (record != null) {
    		editPane.addPaletteNodesFromXML(record.getAttribute("layout"));
    		editPane.show();
    		editToolbar.show();
    		showPalette();
    	}
    	currentRecord = record;
    }

    private void viewDashboard() {
    	ListGridRecord record = dashboardsList.getSelectedRecord();
    	if (record != null) {
    		editPane.addPaletteNodesFromXML(record.getAttribute("layout"));
    		editPane.show();
    		editToolbar.hide();
    		hidePalette();
    	}
    	currentRecord = record;
    }

    private void newDashboard() {
        clearCurrentDashboard();
        currentRecord = null;

        // Add a PortalLayout to the editPane
        editPane.addFromPaletteNode(initialPortalPaletteNode);
        editPane.show();
        editToolbar.show();
        showPalette();

        saveDashboard();
    }

    private void cloneDashboard() {
    	ListGridRecord record = dashboardsList.getSelectedRecord();
    	if (record != null) {
    		cloneRecord(record);
    	}
    }

    private void showPalette() {
    	selector.enableTab(1);
    	selector.selectTab(1);
    }

    private void hidePalette() {
    	selector.disableTab(1);
    }

    private void refreshDashboard() {
    	clearCurrentDashboard();
    	editDashboard();
    }

    private void saveDashboard() {
		String editNodes = editPane.serializeAllEditNodes();

        if (currentRecord != null) {
            currentRecord.setAttribute("layout", editNodes);
            dashboardsList.updateData(currentRecord);
        } else {
        	ListGridRecord newRecord = new ListGridRecord();
        	newRecord.setAttribute("description", "New dashboard");
        	newRecord.setAttribute("layout", editNodes);

        	dashboardsList.addData(newRecord, new DSCallback() {
    			@Override
    			public void execute(DSResponse dsResponse, Object data, DSRequest dsRequest) {
    				Record[] records = dsResponse.getData();
    				if (records != null) {
    					dashboardsList.selectSingleRecord(records[0]);
                        currentRecord = records[0];
    				}
    			}
    		});
        }
    }

    private void cloneRecord(ListGridRecord record) {
    	ListGridRecord newRecord = new ListGridRecord();
    	newRecord.setAttribute("description", record.getAttribute("description"));
    	newRecord.setAttribute("layout", record.getAttribute("layout"));
    	
    	dashboardsList.addData(newRecord);
    }

    public String getIntro() {
        return DESCRIPTION;
    }

    public SourceEntity[] getSourceUrls() {
        return new SourceEntity[]{
            new SourceEntity("PortalGrid.java", JAVA, "source/portal/PortalGrid.java.html", false),
        };
    }
}
